#include <nan.h>
#include "Werapi.h"
#include <atlstr.h>
#include <Windows.h>
#include <eh.h>

using namespace Nan;

using v8::Array;
using v8::FunctionTemplate;
using v8::Handle;
using v8::Local;
using v8::Object;
using v8::String;
using v8::Value;

void  throwWatsonException(char *message);
HRESULT getString(Value* val, WCHAR* buffer, int bufferlen);
Local<String> getJSString(char* input);
HRESULT getStringProp(Local<Object> obj, char* propName, WCHAR* buffer, int bufferlen);
int getIntProp(Local<Object> obj, char* propName);
bool getBoolProp(Local<Object> obj, char* propName);


void createReport(const Nan::FunctionCallbackInfo<Value>& args) {
	// The JS call looks like this:
	// CreateReport(eventType: string, reportType: WerReportType, files: FileInfo[], params : WerParams, consent: WerConsent, submitFlags: number)

	WCHAR eventType[256];
	// extract what I need from args to get values for
	HRESULT eventStringResult = getString(*args[0], eventType, _countof(eventType));
	if (!SUCCEEDED(eventStringResult)) {
		throwWatsonException("Invalid EventType");
		return;
	}
	WER_REPORT_TYPE repType = static_cast<WER_REPORT_TYPE>(args[1]->Int32Value());
	// Not strictly needed.
	//PWER_REPORT_INFORMATION ReportInformation;

#if DEBUG
	printf("\neventType: %ls", eventType);
	printf("\nreportType: %i", repType);
#endif

	HREPORT hReportHandle;

	//call the first API
	HRESULT result = WerReportCreate(eventType, repType, NULL/*ReportInformation*/, &hReportHandle);
	if (!SUCCEEDED(result)) {
		throwWatsonException("WerCreateReport did not succeed");
		return;
	}

#if DEBUG
	printf("\nHRESULT: %i", result);
#endif

	// get each file info out of the array and call WerReportAddFile with it.
	if (args[2]->IsArray()) {
		Handle<Array> files = Handle<Array>::Cast(args[2]);
		for (unsigned int i = 0; i < files->Length(); i++) {
			v8::Local<v8::Object> file = files->Get(i)->ToObject();
			/*
			The File info interface looks like this:
			export interface FileInfo {
				path: string;
				fileType: WerFileType;
				anonymousData: boolean;
				deleteWhenDone: boolean;
			}
			*/
			WCHAR path[256];
			HRESULT pathResult = getStringProp(file, "path", path, _countof(path));
			if (!SUCCEEDED(pathResult)) {
				throwWatsonException("Problem encountered while getting file path");
				return;
			}
			if (wcslen(path) == 0) {
				// File was not present
				continue;
			}
			WER_FILE_TYPE fileType = static_cast<WER_FILE_TYPE>(getIntProp(file, "fileType"));
			bool anonData = getBoolProp(file, "anonymousData");
			bool deleteWhenDone = getBoolProp(file, "deleteWhenDone");
			DWORD flags = (deleteWhenDone ? WER_FILE_DELETE_WHEN_DONE : 0) | (anonData ? WER_FILE_ANONYMOUS_DATA : 0);

#if DEBUG
			printf("\n\nfile number %i", i + 1);
			printf("\nfile path: %ls", path);
			printf("\nFileType: %i", fileType);
			printf("\nanonData: %s", (anonData ? "true" : "false"));
			printf("\ndeleteWhenDone: %s", (deleteWhenDone ? "true" : "false"));
#endif

			// Call the WER API
			HRESULT fileResult = WerReportAddFile(hReportHandle, path, fileType, flags);
			if (!SUCCEEDED(fileResult)) {
				throwWatsonException("WerReportAddFile did not succeed");
				return;
			}
#if DEBUG
			printf("\nFile result: %i", fileResult);
#endif
		}
	}

	//if we have bucket params, set those
	if (args[3]->IsArray()) {
		Handle<Array> params = Handle<Array>::Cast(args[3]);
		for (unsigned int i = 0; i < params->Length(); i++) {
			v8::Local<v8::Object> param = params->Get(i)->ToObject();
			/*
			The WerParam interface looks like this:
			export interface WerParam {
				paramNumber: number;
				paramValue: string;
				paramName? : string;
			}
			*/
			DWORD paramNumber = getIntProp(param, "paramNumber");
			WCHAR paramValue[256];
			HRESULT paramValueResult = getStringProp(param, "paramValue", paramValue, _countof(paramValue));
			if (!SUCCEEDED(paramValueResult)) {
				throwWatsonException("Invalid parameter value.");
				return;
			}
			WCHAR paramName[256];
			HRESULT paramNameResult = getStringProp(param, "paramName", paramName, _countof(paramName));
			if (!SUCCEEDED(paramNameResult)) {
				throwWatsonException("Invalid parameter name.");
				return;
			}

#if DEBUG
			printf("\n\nparam number: %i", paramNumber);
			printf("\nparamValue: %ls", paramValue);
			printf("\nparamName: %ls", paramName);
#endif

			HRESULT paramResult = WerReportSetParameter(hReportHandle, paramNumber, paramName, paramValue);
			if (!SUCCEEDED(paramResult)) {
				throwWatsonException("WerReportSetParameter did not succeed");
				return;
			}
#if DEBUG
			printf("\nparamResult: %i", paramResult);
#endif
		}
	}

	//get the consent and submission flags
	WER_CONSENT consent = static_cast<WER_CONSENT>(args[4]->Int32Value());
	DWORD submitFlags = args[5]->Int32Value();

#if DEBUG
	printf("\n\nconsent: %i", consent);
	printf("\nsubmitFlags: %i", submitFlags);
#endif


	//finally, submit the report
	WER_SUBMIT_RESULT submitResult;
	HRESULT submitResult1 = WerReportSubmit(hReportHandle, consent, submitFlags, &submitResult);
	if (!SUCCEEDED(submitResult1)) {
		throwWatsonException("WerReportSubmit did not succeed");
		return;
	}
#if DEBUG
	printf("\n\nsubmit returned result: %i", submitResult);
	printf("\n\nsubmission result: %i", submitResult1);
#endif

	//and clear the handle
	HRESULT closeResult = WerReportCloseHandle(hReportHandle);
	if (!SUCCEEDED(closeResult)) {
		throwWatsonException("WerReportCloseHandle did not succeed");
		return;
	}

#if DEBUG
		printf("\n\nclose result: %i", closeResult);
#endif
}

NAN_MODULE_INIT(InitAll) {
	Nan::Set(target, Nan::New("createReport").ToLocalChecked(),
		Nan::GetFunction(Nan::New<FunctionTemplate>(createReport)).ToLocalChecked());
}

NODE_MODULE(NativeExtension, InitAll)

// Helper methods
//Helper method to take a v8 value and spit out a wchar string
HRESULT getString(Value* val, WCHAR* buffer, int bufferlen) {
	if (val->IsString()) {
		char* temp = *Nan::Utf8String(val->ToString());
		if (strlen(temp) >  (unsigned int) bufferlen) {
			return -2; // input string is too long
		}
		CA2W pszW(temp);
		wcscpy(buffer, pszW.m_szBuffer);
		return 0; //success
	}
	// value is not a string
	return -1;
}

Local<String> getJSString(char* input) {
	MaybeLocal<String> maybe = Nan::New<String>(input);
	Local<String> local;
	maybe.ToLocal(&local);
	return local;
}

// These methods have undefined behavior if the property name given doesn't exist.
// TODO do something about that.
HRESULT getStringProp(Local<Object> obj, char* propName, WCHAR* buffer, int bufferlen) {
	return getString(*(obj->Get(getJSString(propName))), buffer, bufferlen);
}

int getIntProp(Local<Object> obj, char* propName) {
	//TODO use Nan APIs to make this more compatible accross various versions
	return (int)(obj->Get(getJSString(propName))->NumberValue());
}

bool getBoolProp(Local<Object> obj, char* propName) {
	Local<v8::String> propNameInternal = getJSString(propName);
	if (Nan::Has(obj, propNameInternal).FromMaybe(false)) {
		Nan::Maybe<bool> ml = Nan::To<bool>(obj->Get(getJSString(propName)));
		bool propValue = ml.FromMaybe(false);
		return propValue;
	}
	// TODO handle the case where the property does not exist
	return NULL;
	//return obj->Get(getJSString(propName))->BooleanValue();
}

void  throwWatsonException(char *message) {
	Nan::ThrowError(message);
	// TODO add more info about these issues.
}